//------------------------------------------------------------------------------
// File: FoodBaker.cs
// Controls the food baker scenario and its tasks
// Author: Matthew Rudge
//------------------------------------------------------------------------------

$Baker::LogQuest        = "Cakey Makey";
$Baker::LogQuota[1]     = "Gather 15 Food.";
$Baker::LogQuota[2]     = "Gather 25 Food.";
$Baker::LogQuota[3]     = "Gather 35 Food.";
$Baker::LogQuota[4]     = "Gather 45 Food.";
$Baker::LogQuota[5]     = "Gather 50 Food.";
$Baker::LogQuotaLong[1] = "Collect 15 bags of flour so that Jillian can fill her orders.";
$Baker::LogQuotaLong[2] = "Collect 25 bags of flour so that Jillian can bake enough cakes.";
$Baker::LogQuotaLong[3] = "Collect 35 bags of flour to help Jillian keep up with business.";
$Baker::LogQuotaLong[4] = "Collect 45 bags of flour to keep Jillian headed up in the baking world.";
$Baker::LogQuotaLong[5] = "Collect 50 bags of flour so that Jillian can make the cakes for Harris Pilton's wedding.";
$Baker::LogMin          = "%s minutes left.";
$Baker::LogSec          = "%s seconds left.";
$Baker::LogTask         = "Collect the required amount of food in time for Jillian to make her order.";

$Baker::Quota[1] = 15;
$Baker::Quota[2] = 25;
$Baker::Quota[3] = 35;
$Baker::Quota[4] = 45;
$Baker::Quota[5] = 50;

/******************************************************************************/
/** FOOD BAKER HELPERS
/******************************************************************************/
////////////////////////////////////////////////////////////////////////////////
//! Activates the Bake Off tasks
////////////////////////////////////////////////////////////////////////////////
function startFoodBakerTasks()
{
   InitializeQuestGui();
   tsEnableCommand("BuildCmdData", false);
   tsStartTask("FoodBakerTask", tsGetStartingVertexes("FoodBakerTask"));
   TaskModifier.strMarkVertex("FoodBakerTask", "FoodBakerStartDialog", $TSV_AVAIL);
}
function FoodBakerStartDialog::onExit()
{
   tsSendTaskPopUpDlg("ID_TASKNAME_FOODBAKE", "id_fb_intro", $TaskButton::More, "herocharacter");
}
////////////////////////////////////////////////////////////////////////////////
//! Deactivates white or wong tasks
////////////////////////////////////////////////////////////////////////////////
function endFoodBakerTasks()
{
   tsEndTask("FoodBakerTask");
}
// Rewards Player for completing Food Baker Task
function winFoodBakerTask()
{
   tsModifyResource($Resource::Experience, 50, true);
   tsModifyResource($Resource::wood, 200, true);
   tsModifyResource($Resource::gold, 200, true);
   SendProductionToClient(HeroCharacter, "50 experience");
   SendProductionToClient(HeroCharacter, "200 gold");
   SendProductionToClient(HeroCharacter, "200 wood");
   //Quest log
   tsCompleteTaskInTracker("Cakey Makey", "Help Jillian complete her order.");
   //Delete Jill
   jillevans.deleteobject();
}
////////////////////////////////////////////////////////////////////////////////
//! Resets and completes the log vertexes so that the log will properly show
//! time left for quota
////////////////////////////////////////////////////////////////////////////////
function foodBakerResetLogVertexes()
{
   %quota = foodBakerGetQuotaNumber();
   
   // Reset and complete vertexes
   if(%quota > 4) {
      TaskModifier.strResetVertex("FoodBakerTask", "FoodBakerSixMin");
   }
   if(%quota > 3) {
      TaskModifier.strResetVertex("FoodBakerTask", "FoodBakerFiveMin");
   }
   if(%quota > 2) {
      TaskModifier.strResetVertex("FoodBakerTask", "FoodBakerFourMin");
   }
   if(%quota > 1) {
      TaskModifier.strResetVertex("FoodBakerTask", "FoodBakerTwoMin");
   }
   TaskModifier.strResetVertex("FoodBakerTask", "FoodBakerOneMin");
   TaskModifier.strResetVertex("FoodBakerTask", "FoodBakerThirtySec");
   TaskModifier.strResetVertex("FoodBakerTask", "FoodBakerTenSec");
   TaskModifier.strResetVertex("FoodBakerTask", "FoodBakerFiveSec");
   
   // Complete vertexes
   TaskModifier.strMarkVertex("FoodBakerTask", "FoodBakerSixMin", $TSV_AVAIL);
   if(%quota < 5) {
      TaskModifier.strMarkVertex("FoodBakerTask", "FoodBakerSixMin", $TSV_CMPLETE);
   }
   if(%quota < 4) {
      TaskModifier.strMarkVertex("FoodBakerTask", "FoodBakerFiveMin", $TSV_CMPLETE);
   }
   if(%quota < 3) {
      TaskModifier.strMarkVertex("FoodBakerTask", "FoodBakerFourMin", $TSV_CMPLETE);
   }
   if( %quota < 2) {
      TaskModifier.strMarkVertex("FoodBakerTask", "FoodBakerTwoMin", $TSV_CMPLETE);
   }
}
////////////////////////////////////////////////////////////////////////////////
//! Checks to see if quota has been met. If so resources are deducted, and if 
//! not, the proper vertexes are reactivated. Jill responds appropriately
//! \param %vertex Current vertex quota
//! \param %prevertex Vertex that comes before %vertex
////////////////////////////////////////////////////////////////////////////////
function foodBakerQuotaCheck(%vertex, %prevertex)
{
   // Get food count
   %count = rsGetResourceCount($Resource::Food);
   %quota = foodBakerGetQuotaNumber();
   %faildialog = "id_fb_failquota" @ %quota;
   %windialog  = "id_fb_succeedquota" @ %quota;
   %quotacount = $Baker::Quota[%quota];
      
   // Fail case
   if(%count < %quotacount) {
      tsSendTaskPopUpDlg("ID_TASKNAME_FOODBAKE", %faildialog, $TaskButton::Okay, "jill");
      TaskModifier.strResetVertex("FoodBakerTask", %vertex);
      tsActivateVertexes("FoodBakerTask", %prevertex);
      FoodBakerLose.incCompleteCount();
   }
   // Success case
   else {
      tsSendTaskPopUpDlg("ID_TASKNAME_FOODBAKE", %windialog, $TaskButton::Okay, "jill", %windialog);
      tsModifyResource($Resource::Food, %quotacount, false);
   }
}
////////////////////////////////////////////////////////////////////////////////
//! Returns the current quota that the player is working on (1-5)
////////////////////////////////////////////////////////////////////////////////
function foodBakerGetQuotaNumber()
{
   // No more quotas
   if(FoodBakerFifthQuota.hasBeenCompleted()) {
      return -1;
   }
   // On fifth quota
   else if(FoodBakerFourthQuota.hasBeenCompleted()) {
      return 5;
   }
   // On fourth quota
   else if(FoodBakerThirdQuota.hasBeenCompleted()) {
      return 4;
   }
   // On third quota
   else if(FoodBakerSecondQuota.hasBeenCompleted()) {
      return 3;
   }
   // On second quota
   else if(FoodBakerFirstQuota.hasBeenCompleted()) {
      return 2;
   }
   // On first quota
   else if(FoodBakerIntro.hasBeenCompleted()) {
      return 1;
   }
   // Not on quota
   else {
      return -1;
   }
}
////////////////////////////////////////////////////////////////////////////////
//! Returns the time limit for the quota number
//! \param %quota Quota number
////////////////////////////////////////////////////////////////////////////////
function foodBakerGetQuotaTime(%quota)
{
   // Fifth quota, 6 minutes 360
   if(%quota > 4) {
      return 6;
   }
   // Fourth quota, 5 minutes 300
   if(%quota > 3) {
      return 5;
   }
   // Third quota, 4 minutes 240 secs
   if(%quota > 2) {
      return 4;
   }
   // Second quota, 3 minutes 180 secs
   if(%quota > 1) {
      return 3;
   }
   // First quota, 1 minute: 60 secs
   if(%quota > 0) {
      return 60;
   }
   // Not a quota, return 0 mins
   else {
      return 0;
   }
}
////////////////////////////////////////////////////////////////////////////////
//! Starts the disasters for the food bake-off. Includes famine, plague, and 
//! drought
////////////////////////////////////////////////////////////////////////////////
function startFoodBakerDisasters()
{
   %scenario = new SLScenario() 
   {
      stack = true;
      disasterPlagueRate  = "20 80";
      disasterFamineRate  = "30 80";
      disasterDroughtRate = "20 80";
   };
   
   // Update the scenario
   %scenario.disasterPlagueScenario  = createPlagueDisasterScenario(40, 30, 0.5);
   %scenario.disasterFamineScenario  = createFamineDisasterScenario(10, 0, 0.50, $Resource::Food);
   %scenario.disasterDroughtScenario = createDroughtDisasterScenario(10, 45, 0.9);
   
   %scenario.disasterPlagueScenario.addProduce = "wheat";
   %scenario.disasterPlagueScenario.addProduce = "corn";
   %scenario.disasterPlagueScenario.addProduce = "pumpkin";
   %scenario.disasterPlagueScenario.addProduce = "chicken";
   %scenario.disasterPlagueScenario.addProduce = "pig";
   %scenario.disasterPlagueScenario.addProduce = "cow";
   %scenario.disasterPlagueScenario.addRequirement = Farm;
   %scenario.disasterPlagueScenario.addRequirement = Ranch;
   %scenario.disasterDroughtScenario.addRequirement = well;
   
   LoadScenario(%scenario);
}

/******************************************************************************/
/** FOOD BAKER GAME
/******************************************************************************/

//-FoodBakerIntro Vertex--------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Gives intro dialog to start bake game once hero enters zone. Spawns first
// civilians to help hero and initializes the quest log
////////////////////////////////////////////////////////////////////////////////
function FoodBakerIntro::onEnter(%this)
{
   Parent::onEnter(%this);
   
   // Send dialog between baker and hero
   tsSendTaskPopUpDlg("ID_TASKNAME_FOODBAKE", "id_fb_onzone1", $TaskButton::More, "jill");
   tsSendTaskPopUpDlg("ID_TASKNAME_FOODBAKE", "id_fb_onzone2", $TaskButton::More, "HeroCharacter");
   tsSendTaskPopUpDlg("ID_TASKNAME_FOODBAKE", "id_fb_begin", $TaskButton::Okay, "jill", "id_fb_begin");
}
function FoodBakerIntro::onExit(%this)
{
   Parent::onExit(%this);
   tsEnableCommand("BuildCmdData", true);
   // Start quest log and timer
   tsAddQuestToTracker(true, $Baker::LogQuest, "", "Help Jillian keep her bakery supplied while she whips up delicious cakes for her customers.", "Reward: 200 Gold, 200 Wood and 50 Experience.");
   tsReadyQuest($Baker::LogQuest);
   %brief = slgFormatUIString($Baker::LogSec, 5);
   tsAddTaskToTracker($Baker::LogQuest, %brief, $Baker::LogTask);
   tsAddTaskToTracker($Baker::LogQuest, $Baker::LogQuota[1], $Baker::LogQuotaLong[1]);
   
   // Create two civs to help
   tsCreateObj("MaleCiv", "", $OST_PLAYER, jill.getPosition());
   tsCreateObj("FemaleCiv", "", $OST_PLAYER, jill.getPosition());
   tsConvertObjTeam("bakecabin", $OST_PLAYER);
   // Start disasters
   startFoodBakerDisasters();
}
//-FoodBakerPreFirstQuota Vertex------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Resets the quest log
////////////////////////////////////////////////////////////////////////////////
function FoodBakerPreFirstQuota::onUpdate(%this)
{
   Parent::onUpdate(%this);
   
   // Set log
   foodBakerResetLogVertexes();
   %old = slgFormatUIString($Baker::LogSec, 5);
   %new = slgFormatUIString($Baker::LogSec, foodBakerGetQuotaTime(1));
   tsReplaceTaskInTracker($Baker::LogQuest, %old, %new, $Baker::LogTask);
   
   // Finish setup
   tsCompleteVertexes("FoodBakerTask", "FoodBakerPreFirstQuota"); 
}
//-FoodBakerFirstQuota Vertex---------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// At end of quota, win or lose conditions are evaluated. Sets log for next 
// quota
////////////////////////////////////////////////////////////////////////////////
function FoodBakerFirstQuota::onMessage(%this, %msg, %params)
{
   %complete = %this.isFunctionComplete($TSF_TIMERFIRE, 0);
   Parent::onMessage(%this, %msg, %params);
   if(!%complete && %this.isFunctionComplete($TSF_TIMERFIRE, 0)) {
      foodBakerQuotaCheck("FoodBakerFirstQuota", "FoodBakerPreFirstQuota");
   }
}
function FoodBakerFirstQuota::onExit(%this)
{
   Parent::onExit(%this);

   // Complete quota in log and set next one
   tsCompleteTaskInTracker($Baker::LogQuest, $Baker::LogQuota[1]);
   tsAddTaskToTracker($Baker::LogQuest, $Baker::LogQuota[2], $Baker::LogQuotaLong[2]);
}
//-FoodBakerPreSecondQuota Vertex-----------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Resets the quest log
////////////////////////////////////////////////////////////////////////////////
function FoodBakerPreSecondQuota::onUpdate(%this)
{
   Parent::onUpdate(%this);
   
   // Set log
   foodBakerResetLogVertexes();
   %old = slgFormatUIString($Baker::LogSec, 5);
   %new = slgFormatUIString($Baker::LogMin, foodBakerGetQuotaTime(2));
   tsReplaceTaskInTracker($Baker::LogQuest, %old, %new, $Baker::LogTask);
   
   // Finish setup
   tsCompleteVertexes("FoodBakerTask", "FoodBakerPreSecondQuota");
}
//-FoodBakerSecondQuota Vertex--------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// At end of quota, win or lose conditions are evaluated. Sets log for next 
// quota
////////////////////////////////////////////////////////////////////////////////
function FoodBakerSecondQuota::onMessage(%this, %msg, %params)
{
   %complete = %this.isFunctionComplete($TSF_TIMERFIRE, 0);
   Parent::onMessage(%this, %msg, %params);
   if(!%complete && %this.isFunctionComplete($TSF_TIMERFIRE, 0)) {
      foodBakerQuotaCheck("FoodBakerSecondQuota", "FoodBakerPreSecondQuota");
   }
}
function FoodBakerSecondQuota::onExit(%this)
{
   Parent::onExit(%this);
   
   // Complete quota in log and set next one
   tsCompleteTaskInTracker($Baker::LogQuest, $Baker::LogQuota[2]);
   tsAddTaskToTracker($Baker::LogQuest, $Baker::LogQuota[3], $Baker::LogQuotaLong[3]);
   
   // Create three more civs to help
   tsCreateObj("MaleCiv", "", $OST_PLAYER, jill.getPosition());
   tsCreateObj("FemaleCiv", "", $OST_PLAYER, jill.getPosition());
   tsCreateObj("FemaleCiv", "", $OST_PLAYER, jill.getPosition());
}
//-FoodBakerPreThirdQuota Vertex------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Resets the quest log
////////////////////////////////////////////////////////////////////////////////
function FoodBakerPreThirdQuota::onUpdate(%this)
{
   Parent::onUpdate(%this);
   
   // Set log
   foodBakerResetLogVertexes();
   %old = slgFormatUIString($Baker::LogSec, 5);
   %new = slgFormatUIString($Baker::LogMin, foodBakerGetQuotaTime(3));
   tsReplaceTaskInTracker($Baker::LogQuest, %old, %new, $Baker::LogTask);
   
   // Finish setup
   tsCompleteVertexes("FoodBakerTask", "FoodBakerPreThirdQuota");
}
//-FoodBakerThirdQuota Vertex---------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// At end of quota, win or lose conditions are evaluated. Sets log for next 
// quota
////////////////////////////////////////////////////////////////////////////////
function FoodBakerThirdQuota::onMessage(%this, %msg, %params)
{
   %complete = %this.isFunctionComplete($TSF_TIMERFIRE, 0);
   Parent::onMessage(%this, %msg, %params);
   if(!%complete && %this.isFunctionComplete($TSF_TIMERFIRE, 0)) {
      foodBakerQuotaCheck("FoodBakerThirdQuota", "FoodBakerPreThirdQuota");
   }
}
function FoodBakerThirdQuota::onExit(%this)
{
   Parent::onExit(%this);
   
   // Complete quota in log and set next one
   tsCompleteTaskInTracker($Baker::LogQuest, $Baker::LogQuota[3]);
   tsAddTaskToTracker($Baker::LogQuest, $Baker::LogQuota[4], $Baker::LogQuotaLong[4]);
}
//-FoodBakerPreFourthQuota Vertex-----------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Resets the quest log
////////////////////////////////////////////////////////////////////////////////
function FoodBakerPreFourthQuota::onUpdate(%this)
{
   Parent::onUpdate(%this);
   
   // Set log
   foodBakerResetLogVertexes();
   %old = slgFormatUIString($Baker::LogSec, 5);
   %new = slgFormatUIString($Baker::LogMin, foodBakerGetQuotaTime(4));
   tsReplaceTaskInTracker($Baker::LogQuest, %old, %new, $Baker::LogTask);
   
   // Finish setup
   tsCompleteVertexes("FoodBakerTask", "FoodBakerPreFourthQuota");
}
//-FoodBakerFourthQuota Vertex--------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// At end of quota, win or lose conditions are evaluated. Sets log for next 
// quota
////////////////////////////////////////////////////////////////////////////////
function FoodBakerFourthQuota::onMessage(%this, %msg, %params)
{
   %complete = %this.isFunctionComplete($TSF_TIMERFIRE, 0);
   Parent::onMessage(%this, %msg, %params);
   if(!%complete && %this.isFunctionComplete($TSF_TIMERFIRE, 0)) {
      foodBakerQuotaCheck("FoodBakerFourthQuota", "FoodBakerPreFourthQuota");
   }
}
function FoodBakerFourthQuota::onExit(%this)
{
   Parent::onExit(%this);
   
   // Complete quota in log and set next one
   tsCompleteTaskInTracker($Baker::LogQuest, $Baker::LogQuota[4]);
   tsAddTaskToTracker($Baker::LogQuest, $Baker::LogQuota[5], $Baker::LogQuotaLong[5]);
}
//-FoodBakerPreFifthQuota Vertex------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Resets the quest log
////////////////////////////////////////////////////////////////////////////////
function FoodBakerPreFifthQuota::onUpdate(%this)
{
   Parent::onUpdate(%this);
   
   // Set log
   foodBakerResetLogVertexes();
   %old = slgFormatUIString($Baker::LogSec, 5);
   %new = slgFormatUIString($Baker::LogMin, foodBakerGetQuotaTime(5));
   tsReplaceTaskInTracker($Baker::LogQuest, %old, %new, $Baker::LogTask);
   
   // Finish setup
   tsCompleteVertexes("FoodBakerTask", "FoodBakerPreFifthQuota");
}
//-FoodBakerFifthQuota Vertex---------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// At end of quota, win or lose conditions are evaluated. Completes log
////////////////////////////////////////////////////////////////////////////////
function FoodBakerFifthQuota::onMessage(%this, %msg, %params)
{
   %complete = %this.isFunctionComplete($TSF_TIMERFIRE, 0);
   Parent::onMessage(%this, %msg, %params);
   if(!%complete && %this.isFunctionComplete($TSF_TIMERFIRE, 0)) {
      foodBakerQuotaCheck("FoodBakerFifthQuota", "FoodBakerPreFifthQuota");
   }
}
function FoodBakerFifthQuota::onExit(%this)
{
   Parent::onExit(%this);
   
   // Complete quota in log
   tsCompleteTaskInTracker($Baker::LogQuest, $Baker::LogQuota[5]);
}
//-FoodBakerWin Vertex----------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Pops dialog on successful completion of all quotas. Takes hero back to main
// map
////////////////////////////////////////////////////////////////////////////////
function FoodBakerWin::onEnter(%this)
{
   tsSendTaskPopUpDlg("ID_TASKNAME_FOODBAKE", "id_fb_win", $TaskButton::Okay, "jill", "id_fb_win");
}
function FoodBakerWin::onExit(%this)
{
   %scenario = new SLScenario() {
      stack = false;
   };
   LoadScenario(%scenario, endFoodBakerTasks, winFoodBakerTask);
   
   // complete the bake off satellite
   mapsatellite_bakeoff.disabled = false;
}
//-FoodBakerLose Vertex---------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Ends task on lose condition (no more retries)
////////////////////////////////////////////////////////////////////////////////
function FoodBakerLose::onUpdate(%this)
{
   Parent::onUpdate(%this);
   
   // Shut down quest
   if(%this.getCurrentCompleteCount() >= %this.getCompleteCount()) {
      TaskModifier.strMarkVertex("FoodBakerTask", "FoodBakerPreFirstQuota", $TSV_UNAVAIL);
      TaskModifier.strMarkVertex("FoodBakerTask", "FoodBakerSixMin", $TSV_UNAVAIL);
      tsCompleteVertexes("FoodBakerTask", "FoodBakerLose");
      tsSendTaskPopUpDlg("ID_TASKNAME_FOODBAKE", "id_fb_fail", $TaskButton::Okay, "jill", "id_fb_fail");
   }
}
//-FoodBakerRestart Vertex------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Shows restart dialog
////////////////////////////////////////////////////////////////////////////////
function FoodBakerRestart::onExit(%this)
{
   Parent::onExit(%this);
   tsRestartScenario();
}

/******************************************************************************/
/** FOOD BAKER LOG VERTEXES
/******************************************************************************/
//-FoodBakerTenMin Vertex-------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Updates log timer and sends alert message
////////////////////////////////////////////////////////////////////////////////
function FoodBakerSixMin::onExit(%this)
{
   Parent::onExit(%this);
   
   %quota = foodBakerGetQuotaNumber();
   if(%quota < 0) {
      return;
   }
   
   // Log
   %old = slgFormatUIString($Baker::LogMin, 6);
   %new = slgFormatUIString($Baker::LogMin, 5);
   tsReplaceTaskInTracker($Baker::LogQuest, %old, %new, $Baker::LogTask);
   
   // Hud
   %str = slgGetUIString("id_min_timeleft");
   %str = slgFormatUIString(%str, 5);
   tsSendAlertMessage(%str);
}

//-FoodBakerEightMin Vertex-----------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Updates log timer and sends alert message
////////////////////////////////////////////////////////////////////////////////
function FoodBakerFiveMin::onExit(%this)
{
   Parent::onExit(%this);
   
   %quota = foodBakerGetQuotaNumber();
   if(%quota < 0) {
      return;
   }
   
   // Log
   %old = slgFormatUIString($Baker::LogMin, 5);
   %new = slgFormatUIString($Baker::LogMin, 4);
   tsReplaceTaskInTracker($Baker::LogQuest, %old, %new, $Baker::LogTask);
   
   // Hud
   %str = slgGetUIString("id_min_timeleft");
   %str = slgFormatUIString(%str, 4);
   tsSendAlertMessage(%str);
}
//-FoodBakerFiveMin Vertex------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Updates log timer and sends alert message
////////////////////////////////////////////////////////////////////////////////
function FoodBakerFourMin::onExit(%this)
{
   Parent::onExit(%this);
   
   %quota = foodBakerGetQuotaNumber();
   if(%quota < 0) {
      return;
   }
   
   // Log
   %old = slgFormatUIString($Baker::LogMin, 4);
   %new = slgFormatUIString($Baker::LogMin, 3);
   tsReplaceTaskInTracker($Baker::LogQuest, %old, %new, $Baker::LogTask);
   
   // Hud
   %str = slgGetUIString("id_min_timeleft");
   %str = slgFormatUIString(%str, 3);
   tsSendAlertMessage(%str);
}
//-FoodBakerTwoMin Vertex-------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Updates log timer and sends alert message
////////////////////////////////////////////////////////////////////////////////
function FoodBakerTwoMin::onExit(%this)
{
   Parent::onExit(%this);
   
   %quota = foodBakerGetQuotaNumber();
   if(%quota < 0) {
      return;
   }
   
   // Log
   %old = slgFormatUIString($Baker::LogMin, 3);
   %new = slgFormatUIString($Baker::LogSec, 60);
   tsReplaceTaskInTracker($Baker::LogQuest, %old, %new, $Baker::LogTask);
   
   // Hud
   %str = slgGetUIString("id_sec_timeleft");
   %str = slgFormatUIString(%str, 60);
   tsSendAlertMessage(%str);
}
//-FoodBakerOneMin Vertex-------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Updates log timer and sends alert message
////////////////////////////////////////////////////////////////////////////////
function FoodBakerOneMin::onExit(%this)
{
   Parent::onExit(%this);
   
   %quota = foodBakerGetQuotaNumber();
   if(%quota < 0) {
      return;
   }
   
   // Log
   %old = slgFormatUIString($Baker::LogSec, 60);
   %new = slgFormatUIString($Baker::LogSec, 30);
   tsReplaceTaskInTracker($Baker::LogQuest, %old, %new, $Baker::LogTask);
   
   // Hud
   %str = slgGetUIString("id_sec_timeleft");
   %str = slgFormatUIString(%str,  30);
   tsSendAlertMessage(%str);
}
//-FoodBakerThirtySec Vertex----------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Updates log timer and sends alert message
////////////////////////////////////////////////////////////////////////////////
function FoodBakerThirtySec::onExit(%this)
{
   Parent::onExit(%this);
   
   %quota = foodBakerGetQuotaNumber();
   if(%quota < 0) {
      return;
   }
   
   // Log
   %old = slgFormatUIString($Baker::LogSec, 30);
   %new = slgFormatUIString($Baker::LogSec, 10);
   tsReplaceTaskInTracker($Baker::LogQuest, %old, %new, $Baker::LogTask);
   
   // Hud
   %str = slgGetUIString("id_sec_timeleft");
   %str = slgFormatUIString(%str,  10);
   tsSendAlertMessage(%str);
}
//-FoodBakerTenSec Vertex-------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Updates log timer and sends alert message
////////////////////////////////////////////////////////////////////////////////
function FoodBakerTenSec::onExit(%this)
{
   Parent::onExit(%this);
   
   %quota = foodBakerGetQuotaNumber();
   if(%quota < 0) {
      return;
   }
   
   // Log
   %old = slgFormatUIString($Baker::LogSec, 10);
   %new = slgFormatUIString($Baker::LogSec, 5);
   tsReplaceTaskInTracker($Baker::LogQuest, %old, %new, $Baker::LogTask);
   
   // Hud
   %str = slgGetUIString("id_sec_timeleft");
   %str = slgFormatUIString(%str,  5);
   tsSendAlertMessage(%str);
}

// End FoodBaker.cs